/**
* \file: errmemd_backend.c
*
* Definition of the abstraction layer backend.
* 
* This file implements a kind of abstraction to handle concrete implemented
* backends. It's like polymorphismus.
*
* \component: errmemd
*
* \author: Kai Tomerius (ktomerius@de.adit-jv.com)
*          Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/

#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <syslog.h>

#include "errmem_backend.h"
#include "errmem_lib.h"
#include "errmem_socket_interface.h"

//Standard lookup-table for cyclic redundancy code checksum calculation
//with used generator standard polynom (x^16+x^15+x^2+1) = 0x8005

static uint16_t const lk_up_tbl_16[256] = {
	0x0000, 0xC0C1, 0xC181, 0x0140, 0xC301, 0x03C0, 0x0280, 0xC241, 
	0xC601, 0x06C0, 0x0780, 0xC741, 0x0500, 0xC5C1, 0xC481, 0x0440,
	0xCC01, 0x0CC0, 0x0D80, 0xCD41, 0x0F00, 0xCFC1, 0xCE81, 0x0E40,
	0x0A00, 0xCAC1, 0xCB81, 0x0B40, 0xC901, 0x09C0, 0x0880, 0xC841,
	0xD801, 0x18C0, 0x1980, 0xD941, 0x1B00, 0xDBC1, 0xDA81, 0x1A40,
	0x1E00, 0xDEC1, 0xDF81, 0x1F40, 0xDD01, 0x1DC0, 0x1C80, 0xDC41,
	0x1400, 0xD4C1, 0xD581, 0x1540, 0xD701, 0x17C0, 0x1680, 0xD641,
	0xD201, 0x12C0, 0x1380, 0xD341, 0x1100, 0xD1C1, 0xD081, 0x1040,
	0xF001, 0x30C0, 0x3180, 0xF141, 0x3300, 0xF3C1, 0xF281, 0x3240,
	0x3600, 0xF6C1, 0xF781, 0x3740, 0xF501, 0x35C0, 0x3480, 0xF441,
	0x3C00, 0xFCC1, 0xFD81, 0x3D40, 0xFF01, 0x3FC0, 0x3E80, 0xFE41,
	0xFA01, 0x3AC0, 0x3B80, 0xFB41, 0x3900, 0xF9C1, 0xF881, 0x3840,
	0x2800, 0xE8C1, 0xE981, 0x2940, 0xEB01, 0x2BC0, 0x2A80, 0xEA41,
	0xEE01, 0x2EC0, 0x2F80, 0xEF41, 0x2D00, 0xEDC1, 0xEC81, 0x2C40,
	0xE401, 0x24C0, 0x2580, 0xE541, 0x2700, 0xE7C1, 0xE681, 0x2640,
	0x2200, 0xE2C1, 0xE381, 0x2340, 0xE101, 0x21C0, 0x2080, 0xE041,
	0xA001, 0x60C0, 0x6180, 0xA141, 0x6300, 0xA3C1, 0xA281, 0x6240,
	0x6600, 0xA6C1, 0xA781, 0x6740, 0xA501, 0x65C0, 0x6480, 0xA441,
	0x6C00, 0xACC1, 0xAD81, 0x6D40, 0xAF01, 0x6FC0, 0x6E80, 0xAE41,
	0xAA01, 0x6AC0, 0x6B80, 0xAB41, 0x6900, 0xA9C1, 0xA881, 0x6840,
	0x7800, 0xB8C1, 0xB981, 0x7940, 0xBB01, 0x7BC0, 0x7A80, 0xBA41,
	0xBE01, 0x7EC0, 0x7F80, 0xBF41, 0x7D00, 0xBDC1, 0xBC81, 0x7C40,
	0xB401, 0x74C0, 0x7580, 0xB541, 0x7700, 0xB7C1, 0xB681, 0x7640,
	0x7200, 0xB2C1, 0xB381, 0x7340, 0xB101, 0x71C0, 0x7080, 0xB041,
	0x5000, 0x90C1, 0x9181, 0x5140, 0x9301, 0x53C0, 0x5280, 0x9241,
	0x9601, 0x56C0, 0x5780, 0x9741, 0x5500, 0x95C1, 0x9481, 0x5440,
	0x9C01, 0x5CC0, 0x5D80, 0x9D41, 0x5F00, 0x9FC1, 0x9E81, 0x5E40,
	0x5A00, 0x9AC1, 0x9B81, 0x5B40, 0x9901, 0x59C0, 0x5880, 0x9841,
	0x8801, 0x48C0, 0x4980, 0x8941, 0x4B00, 0x8BC1, 0x8A81, 0x4A40,
	0x4E00, 0x8EC1, 0x8F81, 0x4F40, 0x8D01, 0x4DC0, 0x4C80, 0x8C41,
	0x4400, 0x84C1, 0x8581, 0x4540, 0x8701, 0x47C0, 0x4680, 0x8641,
	0x8201, 0x42C0, 0x4380, 0x8341, 0x4100, 0x81C1, 0x8081, 0x4040
};

#define LOOKUP_VALUE(a, b) (lk_up_tbl_16[((a) ^ (b)) & 0xFF])
#define CALC_CRC16(a, b) ( ((a)>>8) ^ (LOOKUP_VALUE(a, b) & 0xFFFF) )

// calculates the crc16 checksum
uint16_t calc_crc16(uint16_t seed, const uint8_t *verify , size_t len)
{
	uint16_t crc = seed;
	while (len--)
		crc = (CALC_CRC16(crc, *verify++) & 0xFFFF);
	return crc;
}

int32_t open_dev(const char* name, int access_mode, int suppress)
{
	int32_t err  = 0;
	int32_t mode = 0;
	if (access_mode & O_RDWR || access_mode & O_WRONLY)
		access_mode |= O_SYNC;
	if (access_mode & O_CREAT)
		mode = S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP;
	if (name) {
		err = open(name, access_mode, mode);
		if (!~err) {
			err = -errno;
			if (access_mode & O_CREAT)
				syslog(LOG_CRIT, "%s %s %s %s", "cannot create device: ",
					   name, " - error = ", strerror(-err));
			else if (err == -ENOENT && !suppress)
				syslog(LOG_CRIT, "%s %s %s %s", "device:",
					   name, "does not exist - error = ", strerror(-err));
			else if (err != -ENOENT)
				syslog(LOG_CRIT, "%s %s %s %s", "cannot open device: ",
					   name, " - error = ", strerror(-err));
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s",
			   "Parameter check failed in function open_dev -> Abort -error = ",
			   strerror(-err));
	}
	return err;
}

int32_t read_dev(int32_t fd, const char* name, char* buf, size_t size)
{
	int32_t err = 0;
	int32_t res = 0;

	if (name && buf && fd >= 0) {
		res = read(fd, buf, size);
		if (!~res)
			err = -errno;
		else if ((res != (int32_t)size)) {
			if (errno)
				err = -errno;
			else
				err = res;
		} else
			err = res;
		if (err < 0)
			syslog(LOG_CRIT, "%s %s %s %u %s %s",
				   "Failed to read from dev = ", name, "read returned: ",
				   res, " - error = ", strerror(-err));
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s", "read_dev: parameter check dev failed");
	}
	return err;
}

int32_t seek_dev(int32_t fd, char* name, off_t pos, int32_t mode)
{
	int32_t err = 0;
    off_t   old = 0;
	off_t   new = 0;

	if (name && pos >= 0 && fd >= 0) {
		if (mode == SEEK_SET) {
			new = lseek(fd, pos, mode);
			if (!~new)
				err = -errno;
			if (!err && (new != pos)) {
				if (errno)
					err = -errno;
				else
					err = -ESPIPE; 
			}
		} else if (mode == SEEK_CUR) {
			old = lseek(fd, 0, SEEK_CUR);
			if (!~old)
				err = errno;
			if (!err) {
				new = lseek(fd, pos, mode);
				if (!~new)
					err = -errno;
				if (!err && (new != (old + pos))) {
					if (errno)
						err = -errno;
					else
						err = -ESPIPE; 
				}
			}
		} else if (mode == SEEK_END) {
			old = lseek(fd, 0, SEEK_END);
			if (!~old)
				err = -errno;
			if (!err) {
				new = lseek(fd, pos, mode);
				if (!~ new)
					err = -errno;
				if (!err && (new != (old + pos))) {
					if (errno)
						err = -errno;
					else
						err = -ESPIPE; 
				}
			}
		} else {
			err = -EINVAL;
			syslog(LOG_CRIT, "%s %s %s %s", "seek_dev (block device): "
				   "device: ", name, " - seek whence unknown - error = ",
				   strerror(-err));
		}
		
		if (err < 0) {
			syslog(LOG_CRIT, "%s %lu %s %s %s %s", "seek_dev (block device): "
				   "can't seek to position: ", old + pos, " dev = ", name,
				   " - error = ", strerror(-err));
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s",
			   "seek_dev (block device): parameter check failed (name, pos)"
			   " - error = ", strerror(-err));
	}

	if (!err)
		err = (int32_t)new;
	return err;
}

int32_t close_dev(int fd, char* name)
{
	int32_t err = 0;
	if (fd >= 0) {
		err = close(fd);
		if (!~err) {
			err = -errno;
			if (name)
				syslog(LOG_CRIT, "%s %s %s %s", "Error closing device: ", name,
					   " - error = ", strerror(-err));
			else
				syslog(LOG_CRIT, "%s %s", "Error closing unknown device"
					   " - error = ", strerror(-err));
		}
	} else {
		err = -EBADF;
		if (name)
			syslog(LOG_CRIT, "%s %s %s %s",
				   "Can't close device: ", name, " - error = ", strerror(-err));
		else
			syslog(LOG_CRIT, "%s %s",
				   "Can't close unknown device - error = ", strerror(-err));
	}
	return err;
}

int32_t write_dev(int32_t fd, char* name, char* buf, size_t size)
{
	int32_t err = 0;
	if (name && buf && size > 0 && fd >= 0) {
		err = write(fd, buf, size);
		if (!~err) {
			err = -errno;
			syslog(LOG_CRIT,"%s %s %s %d %s %p %s %zd %m", "write failed on device ", name,
				   " fd = ", fd, "buf = ", buf, "size = ", size);
		} else if (err >= 0 && (uint32_t)err < size) {
			syslog(LOG_CRIT, "%s %s %s %d %s %zd", "write failed on device ", name, "no space left - "
				   "written bytes = ", err, "bytes to write = ", size);
			err = -ENOSPC;
		} else
			err = 0;
	} else {
		err = -EIO;
		syslog(LOG_CRIT, "%s", "Parameter check in function write_dev failed -> Abort");
	}
	return err;
}

void * get_mem(uint32_t size)
{
	void* p = calloc(1, size);
	if (!p)
		syslog(LOG_CRIT, "%s %d", "cannot allocate memory ", size);
	return p;
}

void errmem_backend_info(ErrmemBackend_t* backend, int fd)
{
	if (backend) {
		if (backend->info)
			backend->info(backend, fd);
		errmem_backend_info(backend->next, fd);
	}
}

void errmem_backend_erase_content(
	ErrmemBackend_t* backend)
{
	if (backend) {
		if (backend->erase)
			backend->erase(backend);
		errmem_backend_erase_content(backend->next);
	}
}

void errmem_backend_destroy(ErrmemBackend_t* backend)
{
	if (backend) {
		ErrmemBackend_t* next = backend->next;
		if (backend->destroy)
			backend->destroy(backend);
		errmem_backend_destroy(next);
	}
}

unsigned errmem_backend_store(
	ErrmemBackend_t** queue, ErrmemBackend_t** rm,
	int len, struct errmem_message* msg)
{
	uint32_t rc  =  0;
	uint32_t err = ~0;
	ErrmemBackend_t*  cur  = *queue;
	ErrmemBackend_t*  nxt  = NULL;
	ErrmemBackend_t** prm  = rm;
	while(cur) {
		nxt = cur->next;
		if (cur->store) {
			rc = cur->store(cur, len, msg);
			if (!~rc) {
				/* In first approach we want to avoid to remove an 
				 * output device from the output queue in case of error 
				 */
				if (cur->get_type) {
					if (!(cur->get_type(cur) & ERRMEM_ALL_OUT)) {
						/* remove persistent backends due to malfunction */
						(void)errmem_remove_backend(queue, cur);
						if (cur->get_name)
							syslog(LOG_CRIT, "%s %s %s", "Backend ",
								   cur->get_name(cur),
								   " no longer accessible due to malfunction !!");
						/* the current default device is removed */
						if (cur->is_default && cur->is_default(cur)) {
							if (*queue && (*queue)->get_name)
								syslog(LOG_CRIT, "%s %s %s", "Backend ",
									   (*queue)->get_name(*queue),
									   " becomes default device !!");
						}
						if (!(*queue))
							syslog(LOG_CRIT, "%s", "No persistent storage is "
								   "yet available!! No more messages will be "
								   "made persistent!!");
						/* add backend to remove queue */
						(void)errmem_add_backend(prm, cur);
						err = ~0;
					}
				} else {
					/* If we can't identify the current device we remove it */
					(void)errmem_remove_backend(queue, cur);
					syslog(LOG_CRIT, "%s", "A backend is no longer accessible "
						   "due to malfunction and removed from queue. !!");
					(void)errmem_add_backend(prm, cur);
					err = ~0;
				}
			} else {
				if (cur->is_default && cur->is_default(cur))
					err = msg->internal.seqnum;
				if (cur->get_type && (cur->get_type(cur) & ERRMEM_ALL_OUT))
					err = msg->internal.seqnum;
			}
		}
		cur = nxt;
	}
	return err;
}

void errmem_backend_create(int32_t acc, ErrmemBackend_t** pbackend)
{
	int32_t err = 0;
	ErrmemBackend_t** pcur = pbackend;
	while (pcur && *pcur) {
		if ((*pcur)->create) {
			err = (*pcur)->create(acc, *pcur);
			if (err < 0) {
				ErrmemBackend_t* b = *pcur;
				*pcur = (*pcur)->next;
				if (*pcur && (*pcur)->set_nr && b->get_nr)
					(*pcur)->set_nr(*pcur, b->get_nr(b));
				if (b->destroy)
					b->destroy(b);
			} else
				pcur = &((*pcur)->next);
		} else
			pcur = &((*pcur)->next);
	}
}

ErrmemBackend_t* errmem_backend_factory(ErrmemBackendDev_t* bdev)
{
	ErrmemBackend_t* backend = NULL;
	if (bdev)
		switch (bdev->type) {
		case BLOCK:
		case BLOCK_RAW:
		case BLOCK_FLASH:
		case BLOCK_FLASH_RAW:
		case BLOCK_NOR:
		case BLOCK_NOR_RAW:
		case BLOCK_NAND:
		case BLOCK_NAND_RAW:
			backend = create_block_structure(bdev);
			break;
		case FILE_FS:
			backend = create_file_structure(bdev);
			break;
		case ERRMEM_DLT_OUT:
			backend = create_dlt_structure(bdev);
			break;
		case ERRMEM_STD_OUT:
		case ERRMEM_STD_ERR:
			backend = create_channel_structure(bdev);
			break;
		case BUFFER:
			backend = errmem_backend_create_buffered(bdev);
			break;
		default:
			break;
		}
	return backend;
}

MessageQueue_t** errmem_get_message(struct msg_header* p, MessageQueue_t** mq)
{
	if (*mq) {
		while ((*mq)->next)
			mq = &(*mq)->next;
		if ((*mq)->msg.length > ERRMEM_MAX_ENTRY_LENGTH)
			(*mq)->msg.length = ERRMEM_MAX_ENTRY_LENGTH;
		if (p)
			p->len = offsetof(struct errmem_message, message) + (*mq)->msg.length + sizeof(*p);
	}
	return mq;
}

void errmem_add_to_mqueue(MessageQueue_t* queue, MessageQueue_t* msg)
{
	if (queue && msg){
		MessageQueue_t** cur = &queue->next;
		while (*cur != NULL)
			cur = &(*cur)->next;
		*cur = msg;
	}
}

void errmem_create_messages(SessionInfo_t *si)
{
	if (si) {
		if (si->msg_queue) {
			MessageQueue_t* m = NULL;
			/* generate additional messages (erase, end, flags) */
			if (si->status < 0) {
				m = calloc(1, sizeof(MessageQueue_t));
				if (m) {
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: READ ERROR MEMORY DEVICE failed for device %s with error: %d\n",
							 si->backend->get_name(si->backend), si->status);
					memcpy(&si->msg_queue->msg.message[0], &m->msg.message, strlen((char*)m->msg.message));
					si->msg_queue->msg.type  = ERRMEM_TYPE_ASCII;
					si->msg_queue->msg.length = strlen((char*)m->msg.message);
					si->msg_queue->next = NULL;
					free(m);
					m = NULL;
				} else
					si->status = -ENOMEM;
			}
			if (si->status >= 0 && si->state == ERRMEM_SESSION_READ_DONE) {
				m = calloc(1, sizeof(MessageQueue_t));
				if (m) {
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH, "= errmemd: ERROR MEMORY READ DONE for device %s\n", si->backend->get_name(si->backend));
					memcpy(&si->msg_queue->msg.message[0], &m->msg.message, strlen((char*)m->msg.message));
					si->msg_queue->msg.type  = ERRMEM_TYPE_ASCII;
					si->msg_queue->msg.length = strlen((char*)m->msg.message);
					si->msg_queue->next = NULL;
					free(m);
					m = NULL;
				} else
					si->status = -ENOMEM;
			}
		    if (si->status >= 0 && si->state == ERRMEM_SESSION_BACKEND_ERASED){
				m = calloc(1, sizeof(MessageQueue_t));
				if (m) {
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: ERROR MEMORY CONTENT of device %s erased\n",
							 si->backend->get_name(si->backend));
					memcpy(&si->msg_queue->msg.message[0], &m->msg.message, strlen((char*)m->msg.message));
					si->msg_queue->msg.type  = ERRMEM_TYPE_ASCII;
					si->msg_queue->msg.length = strlen((char*)m->msg.message);
					si->msg_queue->next = NULL;
					free(m);
					m = NULL;
				} else
					si->status = -ENOMEM;
			}
			
			if (si->status >= 0 && si->msg_queue->msg.internal.flags & ERRMEMD_START){
				m = calloc(1, sizeof(MessageQueue_t));
				if (m)
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: FIRST MESSAGE AFTER DAEMON STARTUP ===========\n");
				else
					si->status = -ENOMEM;
				if (m) {
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					m = NULL;
				}
			}
			if (si->status >= 0 && si->msg_queue->msg.internal.flags & ERRMEMD_SEQ_NR_TWICE){
				m = calloc(1, sizeof(MessageQueue_t));
				if (m)
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: INFO - MESSAGE %d DELIVERED TWICE AND IGNORED ONCE ==\n",
							 si->msg_queue->msg.internal.offset);
				else
					si->status = -ENOMEM;
				if (m) {
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					m = NULL;
				}
			}
			if (si->status >= 0 && si->msg_queue->msg.internal.flags & ERRMEMD_SEQ_NR_MISSMATCH){
				m = calloc(1, sizeof(MessageQueue_t));
				if (m)
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: INFO - MISSMATCH IN SEQUENCE: EXPECTED %d GOT: %d ===\n",
							 si->msg_queue->msg.internal.offset + 1,
							 si->msg_queue->msg.internal.seqnum);
				else
					si->status = -ENOMEM;
				if (m) {
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					m = NULL;
				}
			}
			if (si->status >= 0 && si->msg_queue->msg.internal.flags & ERRMEM_FLAG_ALLOC_FAIL){
				m = calloc(1, sizeof(MessageQueue_t));
				if (m) {
					if (si->msg_queue->msg.internal.offset) {
						snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
								 "= errmemd: INFO - MESSAGE %d HAD ALLOC FAIL. SEQ: %d perhaps may not exist. LAST STORED: %d =\n",
								 si->msg_queue->msg.internal.seqnum,
								 si->msg_queue->msg.internal.seqnum - 1,
								 si->msg_queue->msg.internal.offset);
					}
				}
				else
					si->status = -ENOMEM;
				if (m) {
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					m = NULL;
				}
			}
			if (si->status >= 0 && si->msg_queue->msg.internal.flags & ERRMEMD_MULTIPLE_ALLOC_FAIL){
				m = calloc(1, sizeof(MessageQueue_t));
				if (m)
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: INFO - MESSAGE %d HAD MULTIPLE ALLOC FAIL. SEQ: %d up to %d perhaps may not exist. LAST STORED: %d =\n",
							 si->msg_queue->msg.internal.seqnum,
							 si->msg_queue->msg.internal.offset + 1,
							 si->msg_queue->msg.internal.seqnum - 1,
							 si->msg_queue->msg.internal.offset);
				else
					si->status = -ENOMEM;
				if (m) {
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					m = NULL;
				}
			}
			if (si->status >= 0 && si->msg_queue->msg.internal.flags & ERRMEMD_MESSAGE_MISSING){
				m = calloc(1, sizeof(MessageQueue_t));
				if (m)
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: POTENTIAL ERROR - !!! MISSING MESSAGE(S) - CHECK SEQUENCE !!! CURRENT: %d - LAST STORED: %d =\n",
							 si->msg_queue->msg.internal.seqnum,
							 si->msg_queue->msg.internal.offset);
				else
					si->status = -ENOMEM;
				if (m) {
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					m = NULL;
				}
			}
			if (si->status >= 0 && si->msg_queue->msg.internal.flags & ERRMEM_FLAG_DROPPED){
				m = calloc(1, sizeof(MessageQueue_t));
				if (m)
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: MESSAGES OVERWRITTEN IN RAM =======\n");
				else
					si->status = -ENOMEM;
				if (m) {
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					m = NULL;
				}
			}
			if (si->status >= 0 && si->msg_queue->msg.internal.flags & ERRMEM_FLAG_INCOMPLETE){
				m = calloc(1, sizeof(MessageQueue_t));
				if (m)
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: INCOMPLETE MESSAGE ==========================\n");
				else
					si->status = -ENOMEM;
				if (m) {
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					m = NULL;
				}
			}
			if (si->status >= 0 && si->msg_queue->msg.internal.flags & ERRMEM_FLAG_CRC_ERROR) {
				m = calloc(1, sizeof(MessageQueue_t));
				if (m)
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: MESSAGE CRC ERROR ==========================\n");
				else
					si->status = -ENOMEM;
				if (m) {
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					m = NULL;
				}			
			}
		    if (si->status >= 0 && ~si->seq_new) {
				if (!si->seq_new && !~si->seq_last)
					si->seq_last = 0;
				m = calloc(1, sizeof(MessageQueue_t));
				if (m) {
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: ERRMEM beginning of block %d from device %s\n",
							 si->seq_new, si->backend->get_name(si->backend));
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					si->seq_new = ~0;
					m = NULL;
				} else
					si->status = -ENOMEM;
			}
		    if (si->status >= 0 && si->seq_next > 0) {
				/* check overwrite in case that not the Oth block is concerned */
				int32_t ov = 0;
				if (!~si->seq_last)
					ov = si->seq_next;
				else
					ov = si->seq_next - si->seq_last;
				if (ov > 1 || (ov > 0 && !~si->seq_last)) {
					m = calloc(1, sizeof(MessageQueue_t));
					if (m) {
						if (~si->seq_last && (si->seq_last || 
											   (!si->seq_last && si->persistent)))
							ov--;
						snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
								 "= errmemd: ERRMEM %d blocks are missing in device %s\n",
								 ov, si->backend->get_name(si->backend));
						m->msg.type  = ERRMEM_TYPE_ASCII;
						m->msg.length = strlen((char*)&m->msg.message[0]);
						errmem_add_to_mqueue(si->msg_queue, m);
						si->seq_last = si->seq_next;
						m = NULL;
					} else
						si->status = -ENOMEM;
				}
				if (!~si->seq_last)
					si->seq_last = si->seq_next;
			}


			if (si->status >= 0 && si->msg_queue->msg.internal.flags & ERRMEM_FLAG_RESTART) {
				m = calloc(1, sizeof(MessageQueue_t));
				if (m) {
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "- errmemd: REBOOT ---------------------------------------\n");
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					m = NULL;
				}
				else
					si->status = -ENOMEM;
			}

			if (si->status >= 0 && si->msg_queue->msg.internal.flags & ERRMEM_FLAG_INITIALIZED){
				m = calloc(1, sizeof(MessageQueue_t));
				if (m) {
					snprintf((char*)&m->msg.message[0], ERRMEM_MAX_ENTRY_LENGTH,
							 "= errmemd: POWER UP =====================================\n");
					m->msg.type  = ERRMEM_TYPE_ASCII;
					m->msg.length = strlen((char*)&m->msg.message[0]);
					errmem_add_to_mqueue(si->msg_queue, m);
					m = NULL;
				}
				else
					si->status = -ENOMEM;
			}

		}
	} else {
		syslog(LOG_CRIT, "%s", "failed to analyse message flags - ErrmemConnect is NULL");
	}
}

/* add a backend to the daemon structure either a mass storage or an output */
int32_t errmem_add_backend(ErrmemBackend_t** queue, ErrmemBackend_t* b)
{
	int32_t err = 0;
	ErrmemBackend_t** cur = queue;
	if (cur && b){
		while (*cur != NULL)
			cur = &(*cur)->next;
		*cur = b;
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", "Cannot add backend to queue\n",
			   strerror(-err));
	}
	return err;
}

/* remove a backend from to the daemon structure */
int32_t errmem_remove_backend(ErrmemBackend_t** queue, ErrmemBackend_t* b) {
	int32_t err = 0;
	ErrmemBackend_t** cur = queue;
    if (cur && b) {
		while ((*cur != b) && (*cur != NULL))
			cur = &((*cur)->next);
		if (*cur) {
			*cur = (*cur)->next;
			if (*cur)
				(*cur)->set_default(*cur, b->is_default(b));
			b->next = NULL;
		} else {
			err = -ENOMEDIUM;
			syslog(LOG_CRIT, "%s %s", "Cannot remove backend from queue\n",
				   strerror(-err));
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", "errmem_remove backend: invalid parameters\n",
			   strerror(-err));
	}
	return err;
}

